//$**************************************************************
//$           Copyright (C) 2011 by L-1 Identity Solutions
//$  Name:        $Workfile: CaptureDlg.cpp $
//$  Author:      $Author: Jmacdonald $
//$
//$  Description:	TPAPI Sample Application capture dialog methods
//$					and image callback function
// 
#include "stdafx.h"
#include "../include/tpapi.h"
#include "TPAPI Sample App.h"
#include "CaptureDlg.h"

// External Functions
extern const char *GetErrorMsg(long);
extern HCURSOR LoadAniCursor(UINT);
extern void SetupPalette(int,RGBQUAD *,int);
extern long SetupBitmap(int,int,int,CBitmap *,unsigned char **);
extern void DrawTransparent(CBitmap *,CDC *,int,int,int,int,COLORREF);
extern CCaptureDlg *captureDlg;
extern HCURSOR hIdxCursor;

// Image Window Sizes
#define IMGW_L		13
#define IMGW_R		742
#define IMGW_W		(IMGW_R - IMGW_L)
#define IMGW_T		38
#define IMGW_B		724
#define IMGW_H		(IMGW_B - IMGW_T)

// Real-Time Quality Analysis Minimums
#define MIN_SCORE                50
#define MAX_SCORE                100
#define MIN_ROLLTIME             100		// Minimum acceptable roll time
#define MIN_COVERAGE             125		// Minimum roll coverage area
#define MIN_COVERAGE_HAND        100		// Minimum roll coverage area - handprint

#define LED_NONE  0
#define LED_LEFT  1
#define LED_MID   2
#define LED_RIGHT 3

struct s_ctypeInfo cTypeInfo[NUM_FNGRS] = {
   CTYPE_ROLL,          LED_NONE,
   CTYPE_SLAP_ONE,      LED_NONE,
   CTYPE_SLAP_FOUR,     LED_NONE,
   CTYPE_HAND,          LED_NONE,
   CTYPE_SLAP_FOUR,     LED_LEFT,
   CTYPE_SLAP_FOUR,     LED_RIGHT,
   CTYPE_SLAP_FOUR,     LED_MID,
   CTYPE_SLAP_ONE,      LED_MID,
   CTYPE_SLAP_ONE,      LED_MID,
   CTYPE_ROLL,          LED_MID,
   CTYPE_ROLL,          LED_LEFT,
   CTYPE_ROLL,          LED_LEFT,
   CTYPE_ROLL,          LED_LEFT,
   CTYPE_ROLL,          LED_LEFT,
   CTYPE_ROLL,          LED_MID,
   CTYPE_ROLL,          LED_RIGHT,
   CTYPE_ROLL,          LED_RIGHT,
   CTYPE_ROLL,          LED_RIGHT,
   CTYPE_ROLL,          LED_RIGHT,
   CTYPE_SLAP_PALM,     LED_NONE,
   CTYPE_SLAP_PALM,     LED_NONE
};

char strFngr[][22] = { 
	"Left Four Fingers",
	"Left Thumb",
	"Left Index",
	"Left Middle",
	"Left Ring",
	"Left Little",
	"Right Four Fingers",
	"Right Thumb",
	"Right Index",
	"Right Middle",
	"Right Ring",
	"Right Little",
	"Left Hand",
	"Left Writers",
	"Left Palm",
	"Right Hand",
	"Right Writers",
	"Right Palm",
	"Both Thumbs"
};

int P2PicResID[NUM_FNGRS-2] = {
	IDB_P2_LFOUR,
	IDB_P2_LTHUMB,
	IDB_P2_LINDEX,
	IDB_P2_LMIDDLE,
	IDB_P2_LRING,
	IDB_P2_LLITTLE,

	IDB_P2_RFOUR,
	IDB_P2_RTHUMB,
	IDB_P2_RINDEX,
	IDB_P2_RMIDDLE,
	IDB_P2_RRING,
	IDB_P2_RLITTLE,

	IDB_P2_LHAND,
	IDB_P2_LWRITERS,
	IDB_P2_LPALM,
	IDB_P2_RHAND,
	IDB_P2_RWRITERS,
	IDB_P2_RPALM,

	IDB_P2_THUMBS
};

//=============================================================
//		ImgCallbk
//			Callback function used by capture process to return
//			capture states and images.
//=============================================================
long __stdcall ImgCallbk(unsigned int hDev, unsigned int CapState, 
               unsigned char *ImageBuf, unsigned short Xsize, 
               unsigned short Ysize, void *pCtxt)
{
	DWORD r = TPAPI_NOERROR;

   CCaptureDlg *capDlg = (CCaptureDlg *)pCtxt;
   capDlg->m_captureState = CapState;
	switch (capDlg->m_captureState)
	{
		case CSTATE_PREVIEW:
				// Display preview image
			if (Xsize > capDlg->XSize) Xsize = capDlg->XSize;
			if (capDlg->m_capType == CTYPE_HAND)
			{		// Ysize is the line number
   			if ((Ysize >= 0) && (Ysize <= capDlg->YSize-BANDED_LINES))
				   for (int i = 0; i < BANDED_LINES; i++)
					   memcpy(&capDlg->bitmapData[((capDlg->YSize-1)
						   - (Ysize+i)) * Xsize], &ImageBuf[i*Xsize], Xsize);
			}
			else if (Ysize <= capDlg->YSize)
				memcpy(capDlg->bitmapData,ImageBuf,Xsize*Ysize);
			capDlg->wImage.Invalidate();
#ifdef DISPLAY_FPS
         capDlg->UpdateDisplay(capDlg->m_captureState);
#endif
			break;

		case CSTATE_STARTED:
				// Capture started, notify operator to lift or roll finger
			capDlg->UpdateDisplay(capDlg->m_captureState);
			break;

		case CSTATE_TOOFAST:
			// Handprint capture refused, notify operator to roll slower
		{
         capDlg->m_bCapturing = false;
			capDlg->UpdateDisplay(capDlg->m_captureState);

			// Query Switches for reScan - wait indefinitely
			unsigned int freq[3] = { 1000, 0, 0 };
			unsigned int freqTime[3] = { 1000, 0, 0 };
			unsigned short swStates = SW_SWITCH_TIMEOUT;

			capDlg->m_queryingSwitches = true;
			TP_QuerySwitches(hDev,QRYSW_CLEAR,0,1,
            freq,freqTime,&swStates);
			capDlg->m_queryingSwitches = false;

			if (swStates & (SW_BSCAN_LATCHED|SW_FSCAN_LATCHED))
				capDlg->PostMessage(WM_USER_PREVIEW,0,0);
			else if (swStates != SW_SWITCH_TIMEOUT)
			   capDlg->PostMessage(WM_CLOSE,0,0);
			break;
		}
		case CSTATE_CAPTURED:
			// Capture complete, analyze RTQA, update display for operator review
			r = capDlg->CallbkCaptured(ImageBuf,Xsize,Ysize);
         capDlg->m_bCapturing = false;
			break;

		case CSTATE_REJECTED:		// SCAN button during Review
			capDlg->UpdateDisplay(capDlg->m_captureState);
			break;

		case CSTATE_FINISHED:
			capDlg->SetUIProgress(capDlg->m_capType, UI_LED_GREEN, FALSE);

			capDlg->m_pImageBuf = (unsigned char *)malloc(Xsize*Ysize);
			if (capDlg->m_pImageBuf != NULL)
			{		// Save finished image & sizes
				memcpy(capDlg->m_pImageBuf,ImageBuf,Xsize*Ysize);
				capDlg->XSize = Xsize;
				capDlg->YSize = Ysize;

					// exit capture window
				capDlg->PostMessage(WM_COMMAND,IDOK,0);
			}
			else capDlg->PostMessage(WM_COMMAND,IDCANCEL,0);
			break;

		case CSTATE_ABORTED:			// SAVE button during Preview
		case CSTATE_CANCELED:		// rsp to StopPreview
			capDlg->SetUIProgress(capDlg->m_capType, UI_LED_RED, TRUE);

			capDlg->XSize = capDlg->YSize = 0;
			capDlg->PostMessage(WM_CLOSE,0,0);
			break;

		case CSTATE_FAILED:
			if (IsWindow(capDlg->m_hWnd))
			{
				capDlg->MessageBox("Hardware or System Error",
					"Capture Failed",MB_ICONERROR);
   			capDlg->XSize = capDlg->YSize = 0;
				capDlg->PostMessage(WM_CLOSE,0,0);
			}
			break;

		default:
			r = TPAPI_INVALIDSTATE;
			break;
	}
	return(r == TPAPI_NOERROR ? 0 : -1);
}

//=====================================================
//		CImageWnd - Fingerprint Image Display Window
//=====================================================

BEGIN_MESSAGE_MAP(CImageWnd, CWnd)
	ON_WM_PAINT()
END_MESSAGE_MAP()

//-----------------------------------------------------
// OnPaint
//		Paint the fingerprint image
//-----------------------------------------------------
void CImageWnd::OnPaint() 
{
	if (!IsIconic())
	{
		CPaintDC dc(this); // device context for painting

		CRect r;
		GetClientRect(&r);

      int XSize = captureDlg->XSize;
      int YSize = captureDlg->YSize;
      int imgXPos = (r.Width() - XSize) / 2;
		int imgYPos = (r.Height() - YSize) / 2;

			// fill background
		COLORREF bkColor = RGB(192,192,192);
		RECT r1 = { 0, 0, imgXPos, r.Height() };
		RECT r2 = { imgXPos+XSize, 0, r.Width(), r.Height() };
		RECT r3 = { imgXPos, 0, imgXPos+XSize, imgYPos };
		RECT r4 = { imgXPos, imgYPos+YSize, imgXPos+XSize, r.Height() };
		dc.FillSolidRect(&r1, bkColor);
		dc.FillSolidRect(&r2, bkColor);
		dc.FillSolidRect(&r3, bkColor);
		dc.FillSolidRect(&r4, bkColor);

			// draw image
      dc.BitBlt(r.left+imgXPos,r.top+imgYPos,XSize,YSize,
         &captureDlg->dcMemImage,0,0,SRCCOPY);
		CWnd::OnPaint();
	}
}

//=====================================================
// CCaptureDlg dialog
//=====================================================
CCaptureDlg::CCaptureDlg(unsigned int hDev,
                         enum eFingerType FingerNum,
								 enum eCaptureType capType,
								 unsigned long capMode,
								 unsigned int RTQAThreshold,
								 CWnd* pParent) 
	: CDialog(CCaptureDlg::IDD, pParent)
{
	m_queryingSwitches = false;
   m_hDev = hDev;
	m_Finger = FingerNum;
	m_capType = capType;
	m_captureMode = capMode;
	m_RTQAThrhld = RTQAThreshold;
	m_pImageBuf = NULL;
   m_hCur = NULL;
	capDsp.stateTxt[0] = 0;
	capDsp.fngrTxt[0] = 0;
	capDsp.operTxt[0] = 0;
   dcMemImage.CreateCompatibleDC(NULL);
}

CCaptureDlg::~CCaptureDlg()
{
	if (m_pImageBuf != NULL) 
      free(m_pImageBuf);

   dcMemImage.DeleteDC();
	m_bmImage.DeleteObject();
	
   capDsp.btns.DeleteObject();
	capDsp.operState.DeleteObject();
	capDsp.fngrImg.DeleteObject();

	m_Font.DeleteObject();
	m_fngrFont.DeleteObject();
	m_stateFont.DeleteObject();
	m_brush.DeleteObject();
	m_bmpAbort.DeleteObject();
	m_bmpScan.DeleteObject();
	m_bmpReScan.DeleteObject();
	m_bmpSave.DeleteObject();
}

BEGIN_MESSAGE_MAP(CCaptureDlg, CDialog)
	ON_WM_CTLCOLOR()
	ON_WM_CLOSE()
	ON_WM_DESTROY()
	ON_BN_CLICKED(IDABORT, OnBnClickedAbort)
	ON_BN_CLICKED(IDSCAN, OnBnClickedScan)
	ON_BN_CLICKED(IDSAVE, OnBnClickedSave)
	ON_MESSAGE(WM_USER_DSPLY, OnUpdateDisplay)
	ON_MESSAGE(WM_USER_PREVIEW, OnStartPreview)
   ON_WM_SETCURSOR()
	ON_WM_TIMER()
END_MESSAGE_MAP()

//---------------------------------------------------------
//		OnInitDialog
//			Initialize dialog controls, start image preview
//---------------------------------------------------------
BOOL CCaptureDlg::OnInitDialog()
{
	long r;
	CDialog::OnInitDialog();

	CButton *b = (CButton *)GetDlgItem(IDABORT);
	m_bmpAbort.DeleteObject();
	m_bmpAbort.LoadBitmap(MAKEINTRESOURCE(IDB_ABORT));
	b->SetBitmap((HBITMAP)m_bmpAbort.GetSafeHandle());

	b = (CButton *)GetDlgItem(IDSCAN);
	m_bmpScan.DeleteObject();
	m_bmpScan.LoadBitmap(MAKEINTRESOURCE(IDB_SCAN));
	b->SetBitmap((HBITMAP)m_bmpScan.GetSafeHandle());
	
	m_bmpReScan.DeleteObject();
	m_bmpReScan.LoadBitmap(MAKEINTRESOURCE(IDB_RESCAN));

	b = (CButton *)GetDlgItem(IDSAVE);
	m_bmpSave.DeleteObject();
	m_bmpSave.LoadBitmap(MAKEINTRESOURCE(IDB_SAVE));
	b->SetBitmap((HBITMAP)m_bmpSave.GetSafeHandle());

	wImage.Create(NULL,NULL,WS_CHILD|WS_VISIBLE,
		CRect(IMGW_L,IMGW_T,IMGW_R,IMGW_B),this,1234);

	m_ctlState = (CStatic *)captureDlg->GetDlgItem(IDC_CSTATE);
	m_ctlFngrTxt = (CStatic *)captureDlg->GetDlgItem(IDC_FNGRTEXT);
	m_ctlFngrImg = (CStatic *)captureDlg->GetDlgItem(IDC_P2);
	m_ctlOperState = (CStatic *)captureDlg->GetDlgItem(IDC_P3);
	m_ctlOperTxt = (CStatic *)captureDlg->GetDlgItem(IDC_P3_SCORE);
	m_ctlBtns = (CStatic *)captureDlg->GetDlgItem(IDC_P4);

	m_brush.CreateSolidBrush(RGB(0,0,128));
	m_stateFont.CreatePointFont(260, "Trebuchet MS Bold");
	m_ctlState->SetFont(&m_stateFont);
	m_fngrFont.CreatePointFont(160, "Trebuchet MS Bold");
	m_ctlFngrTxt->SetFont(&m_fngrFont);
	m_Font.CreatePointFont(180, "Trebuchet MS");
	m_ctlOperTxt->SetFont(&m_Font);

#ifdef DISPLAY_FPS
   m_lblFPS = (CStatic *)captureDlg->GetDlgItem(IDC_FPS);
#endif
#ifdef USE_FSW
   if (m_FootSw.IsOpen()) 
      m_FootSw.CloseDevice();
   if (m_FootSw.OpenDevice() == 0)
      SetTimer(ID_FOOTSW_TIMER,150,NULL);
#endif

		// Start Preview Mode
	HCURSOR hCur = SetCursor(hIdxCursor);
	while ((r = SetPreview()) != TPAPI_NOERROR)
	{
		SetCursor(hCur);
		char errMsg[80];
		if ((r == TPAPI_INVALIDSTATE) || (r == TPAPI_BADRESPONSE)) 
		{
			sprintf(errMsg,"%s\n\nReset Device and Retry?",GetErrorMsg(r));
			if (MessageBox(errMsg,"Start Preview Failed",
				MB_ICONERROR|MB_OKCANCEL) == IDCANCEL) 
			{
				EndDialog(IDCANCEL);
				break;
			}
			else
			{
				hCur = SetCursor(hIdxCursor);
				TP_ResetDevice(m_hDev);
			}
		}
		else
		{
			MessageBox(GetErrorMsg(r),"Start Preview Failed",MB_ICONERROR|MB_OK);
			EndDialog(IDCANCEL);
			break;
		}
	}
	SetCursor(hCur);
   return TRUE;
}

//---------------------------------------------------------
//		OnCtlColor
//			Draw RTQA results pane in correct color
//---------------------------------------------------------
HBRUSH CCaptureDlg::OnCtlColor(CDC* pDC, CWnd* pWnd, UINT nCtlColor) 
{
   // Call the base class implementation first
   HBRUSH hbr = CDialog::OnCtlColor(pDC, pWnd, nCtlColor);

   // Are we painting the correct control?
   int ctlId = pWnd->GetDlgCtrlID();
   if (ctlId == IDC_P3_SCORE)
   {
      // Set the text color
      pDC->SetTextColor(m_crColor);

      // Set the background mode for text to transparent 
      pDC->SetBkMode(TRANSPARENT);

      // Return handle to our CBrush object
      hbr = m_brush;
   }
   else if ((ctlId == IDC_CSTATE) || (ctlId == IDC_FNGRTEXT))
   {
      // Set the text color
      pDC->SetTextColor(RGB(250,250,250));

      // Set the background mode for text to transparent 
      pDC->SetBkMode(TRANSPARENT);

      // Return handle to our CBrush object
      hbr = m_brush;
   }
   return hbr;
}

//---------------------------------------------------------
//		OnSetCursor
//			Prevent Cursor from changing
//---------------------------------------------------------
BOOL CCaptureDlg::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message)
{
   if (m_hCur == NULL)
      return CDialog::OnSetCursor(pWnd, nHitTest, message);
   return true;
}

//-------------------------------------------------------
//		OnTimer
//			Check Foot Switch
//-------------------------------------------------------
void CCaptureDlg::OnTimer(UINT_PTR nIDEvent) 
{
#ifdef USE_FSW
   if (nIDEvent == ID_FOOTSW_TIMER)
   {   // Check Foot Switch
   	if (m_FootSw.IsOpen())
      {
         unsigned int Ports;
   	   if (m_FootSw.ReadPorts(&Ports) == 0)
         {
      	   Ports = ~Ports; // invert to ports values
            if (0x1 & (Ports >> 1))
               OnBnClickedSave();
   	      else if (0x1 & (Ports >> 0) && !m_bCapturing)
               OnBnClickedScan();
         }
			else MessageBox("ReadPorts Failed","Foot Switch"
            " Error",MB_ICONERROR|MB_OK);
      }
   }
#endif
   CDialog::OnTimer(nIDEvent);
}

//---------------------------------------------------------
//		OnCancel - OnClose
//			Prevent ESC key from closing window
//---------------------------------------------------------
void CCaptureDlg::OnCancel() 
{
}

void CCaptureDlg::OnClose()
{
   CDialog::OnCancel();
}

//---------------------------------------------------------
//		OnDestroy
//		   Close FootSwitch device
//---------------------------------------------------------
void CCaptureDlg::OnDestroy()
{
#ifdef USE_FSW
   if (m_FootSw.IsOpen())
   {
      KillTimer(ID_FOOTSW_TIMER);
      m_FootSw.CloseDevice();
   }
#endif
}

//-----------------------------------------------------------
//		CancelQrySwitches
//			Cancel a QrySwitches operation, wait for the blocked 
//       thread to recieved the CANCELED status, and unblock.
//-----------------------------------------------------------
bool CCaptureDlg::CancelQrySwitches(void)
{
   unsigned int freq[3];
   unsigned int freqTime[3];
   unsigned short swStates;
   DWORD r;

   if ((r = TP_QuerySwitches(m_hDev,QRYSW_CANCEL,0,0,freq,
               freqTime,&swStates)) == TPAPI_NOERROR)
   {   // wait for CANCELED QuerySw status to unblock stalled thread
      for (int i = 0; m_queryingSwitches && (i < 30); i++)
         Sleep(50);
   }
   if (m_queryingSwitches)
   {
      char sMsg[128] = { "Query Device Buttons cancel "
                         "operation failed.    \n\n" };
      if (r == TPAPI_NOERROR)
         strcat(sMsg,"No CANCELED status received.\n");
      else sprintf(sMsg,"%s%s\n", sMsg, GetErrorMsg(r));
      MessageBox(sMsg,"Error",MB_ICONERROR);
   }
   return(!m_queryingSwitches);
}

//-----------------------------------------------------------
//		OnBnClickedAbort
//			Abort current capture, close dialog
//-----------------------------------------------------------
void CCaptureDlg::OnBnClickedAbort(void)
{
	HCURSOR hCur = SetCursor(hIdxCursor);
	if (m_queryingSwitches)
	{
      if (CancelQrySwitches())
         PostMessage(WM_CLOSE,0,0);
	}
	else if (TP_CaptureControl(m_hDev,CC_CANCEL_CAPTURE) != TPAPI_NOERROR)
		OnCancel();
	SetCursor(hCur);
}

//-----------------------------------------------------------
//		OnBnClickedScan
//-----------------------------------------------------------
void CCaptureDlg::OnBnClickedScan(void)
{
	if (m_queryingSwitches)
   {
      if (CancelQrySwitches() && (SetPreview() != TPAPI_NOERROR))
      {
         MessageBox("Re-Scan Failed","Error",MB_ICONERROR);
         PostMessage(WM_CLOSE,0,0);
      }
   }
   else 
   {
      EnableGUI(false);
      if (TP_CaptureControl(m_hDev,CC_CAPTURE_IMAGE) == TPAPI_NOERROR)
         m_bCapturing = true;
   }
}

//-----------------------------------------------------------
//		OnBnClickedSave
//-----------------------------------------------------------
void CCaptureDlg::OnBnClickedSave(void)
{
	TP_CaptureControl(m_hDev,CC_FINISH_IMAGE);
}

//-----------------------------------------------------------
//		OnUpdateDisplay
//			Display bitmaps for current capture mode.
//			Signalled from callback routine.
//-----------------------------------------------------------
LRESULT CCaptureDlg::OnUpdateDisplay(WPARAM,LPARAM)
{
   m_ctlState->SetWindowText(capDsp.stateTxt);

      // Operator Guide - Panel 3
	if (capDsp.operTxt[0] != 0)
	{
		m_ctlOperTxt->SetWindowText(capDsp.operTxt);
		m_ctlOperTxt->ShowWindow(SW_SHOW);
	}
	else m_ctlOperTxt->ShowWindow(SW_HIDE);

   if ((m_captureState != CSTATE_CAPTURED) || 
      ((m_captureMode & CMODE_RTQA) == 0))
   {
      capDsp.operStateRes = IDB_P3_BLANK;
      capDsp.operState.DeleteObject();
      capDsp.operState.LoadBitmap(MAKEINTRESOURCE(capDsp.operStateRes));
      m_ctlOperState->SetBitmap((HBITMAP)capDsp.operState.GetSafeHandle());
   }

      // Button Guide - Pane 4
   if(capDsp.btnsRes == IDB_P4_REVIEW)
      GetDlgItem(IDSAVE)->ShowWindow(SW_SHOW);
   else
   	GetDlgItem(IDSAVE)->ShowWindow(SW_HIDE);

	if((capDsp.btnsRes == IDB_P4_REVIEW)||
		(capDsp.btnsRes == IDB_P4_REJECT)||
		(capDsp.btnsRes == IDB_P4_SKIP))
	{
		CButton *b = (CButton *)GetDlgItem(IDSCAN);
		b->SetBitmap((HBITMAP)m_bmpReScan.GetSafeHandle());
	}
	else
	{
		CButton *b = (CButton *)GetDlgItem(IDSCAN);
		b->SetBitmap((HBITMAP)m_bmpScan.GetSafeHandle());
	}

   switch (m_captureState)
	{
		case CSTATE_PREVIEW:
#ifdef DISPLAY_FPS
         if (++numFrms >= 2)
         {
            curTime = GetTickCount();
            if (numFrms == 2) startTime = curTime;
            if (curTime > startTime)
            {
               float fps = ((float)(numFrms - 2) * 1000) / (float)(curTime - startTime);
               char strFPS[32];
               sprintf(strFPS, "%2.2f fps", fps);
               m_lblFPS->SetWindowText(strFPS);
               m_lblFPS->ShowWindow(SW_SHOW);
            }
         }
#endif
			break;

		case CSTATE_STARTED:
         capDsp.btnsRes = IDB_P4_BLANK;
         EnableGUI(false);
			break;

		case CSTATE_CAPTURED:
         if (m_captureMode & CMODE_RTQA)
         {
   	      capDsp.operState.DeleteObject();
            capDsp.operState.LoadBitmap(MAKEINTRESOURCE(capDsp.operStateRes));
	         m_ctlOperState->SetBitmap((HBITMAP)capDsp.operState.GetSafeHandle());
         }
         EnableGUI(true);
			break;

		case CSTATE_REJECTED:		// SCAN button during Review
         EnableGUI(true);
			if (SetPreview() != TPAPI_NOERROR)
				MessageBox("StartPreview Failed","Error",MB_ICONERROR);
			break;

		case CSTATE_FINISHED:
		case CSTATE_ABORTED:			// SAVE button during Preview
      case CSTATE_CANCELED:		// rsp to StopPreview
		case CSTATE_FAILED:
         EnableGUI(true);
         EndDialog(IDOK);    // modal
			break;

		case CSTATE_TOOFAST:
         EnableGUI(true);
         GetDlgItem(IDSAVE)->ShowWindow(SW_HIDE);
			break;

		default:
			break;
	}
   if (m_captureState != CSTATE_PREVIEW) Invalidate();
	return(1);
}

//-----------------------------------------------------------
//		OnStartPreview
//			Start the capture operation.
//-----------------------------------------------------------
LRESULT CCaptureDlg::OnStartPreview(WPARAM,LPARAM)
{
   long r;
	if ((r = SetPreview()) != TPAPI_NOERROR)
   {
      MessageBox(GetErrorMsg(r),"Start Preview Failed",MB_ICONERROR);
      PostMessage(WM_CLOSE,0,0);
   }
	return(1);
}

/////////////////////////////////////////////////////////////////////////////
// CCaptureDlg local methods

//-------------------------------------------------------
//		EnableGUI
//			Enable/disable buttons & busy cursor
//-------------------------------------------------------
void CCaptureDlg::EnableGUI(bool bEnable)
{
   if (bEnable)
   {
      if (m_hCur != NULL)
      { SetCursor(m_hCur); m_hCur = NULL; }
   }
   else if (m_hCur == NULL)
      m_hCur = SetCursor(hIdxCursor);

   int nShow = bEnable ? SW_SHOW : SW_HIDE;
   GetDlgItem(IDSCAN)->ShowWindow(nShow);
   GetDlgItem(IDSAVE)->ShowWindow(nShow);
}

//-----------------------------------------------------------
//    UpdateDisplay
//       Load display bitmaps indicating the capture type, 
//       status, and operator instructions for current 
//       capture mode.
//-----------------------------------------------------------
void CCaptureDlg::UpdateDisplay(int CapState,bool postMsg)
{
   enum eCaptureType capType = cTypeInfo[m_capType].base_ctype;

	switch (CapState)
	{
		case CSTATE_PREVIEW:
			strcpy(capDsp.stateTxt,"PREVIEW");

				// Finger type text/image
			if (capType == CTYPE_ROLL)
				strcpy(capDsp.fngrTxt,"Roll ");
			else strcpy(capDsp.fngrTxt,"Plain ");
			strncat(capDsp.fngrTxt,strFngr[m_Finger-1],24);
         capDsp.fngrImgRes = P2PicResID[m_Finger-1];

				// Instructions & status area
			if (capType == CTYPE_ROLL)
				strcpy(capDsp.operTxt,"CENTER, ROCK,\nthen press\nSCAN");
			else if (capType == CTYPE_SLAP_ONE)
				strcpy(capDsp.operTxt,"CENTER,\nthen press\nSCAN");
			else if (capType == CTYPE_SLAP_FOUR)
				strcpy(capDsp.operTxt,"PLACE FINGERS,\nthen press\nSCAN");
			else if (capType == CTYPE_SLAP_PALM)
				strcpy(capDsp.operTxt,"PLACE PALM,\nthen press\nSCAN");
			else if (capType == CTYPE_HAND)
				strcpy(capDsp.operTxt,"CENTER,\nPLACE HAND,\nthen press\nSCAN");

         capDsp.btnsRes = IDB_P4_PREVIEW;
			break;

		case CSTATE_STARTED:
			strcpy(capDsp.stateTxt,"CAPTURING");

				// Instructions & status area
			if (capType == CTYPE_ROLL)
				strcpy(capDsp.operTxt,"\nROLL FINGER");
			else if (capType == CTYPE_HAND)
				strcpy(capDsp.operTxt,"Pull Hand\ntoward\nSubject");

         capDsp.btnsRes = IDB_P4_CAPTURE;
			break;

		case CSTATE_ABORTED:			// SAVE button during Preview
			strcpy(capDsp.operTxt,"Missing\nfinger?");
			break;

		case CSTATE_TOOFAST:
				// Instructions & status area
			if (capType == CTYPE_HAND)
			{
				m_crColor = RGB(255, 0, 0);	// Display in Red
            capDsp.operStateRes = IDB_P3_REJECT;
				strcpy(capDsp.operTxt,"PULL HAND AT\nA MORE\nMODERATE SPEED");
            capDsp.btnsRes = IDB_P4_SKIP;
			}
		case CSTATE_CAPTURED:
			strcpy(capDsp.stateTxt,"REVIEW");
			break;

		case CSTATE_FINISHED:
			strcpy(capDsp.stateTxt,"SAVING");
			break;

		default:
			break;
	}
		// Update display from dispatch thread
	if (postMsg && IsWindow(m_hWnd))
      PostMessage(WM_USER_DSPLY);
}

//-----------------------------------------------------
//		SetUIProgress
//			Set UI progress LEDs
//-----------------------------------------------------
long CCaptureDlg::SetUIProgress(enum eCaptureType m_capType,
								enum eLEDState LEDState,
								BOOL PreserveFC)
{
	long r = TPAPI_NOERROR;
	struct s_442TriLED UILEDs;
	bool UISupported;

	// set LED
	if ((TP_IsDeviceUISupported(m_hDev,UI_4_4_2_TRI_LED,
      &UISupported) == TPAPI_NOERROR) && UISupported)
	{
		if(PreserveFC)
		{
			if ((r = TP_GetDeviceUI(m_hDev,UI_4_4_2_TRI_LED,
            &UILEDs)) != TPAPI_NOERROR) return(r);
			UILEDs.left_four_progress = UI_LED_OFF;
			UILEDs.right_four_progress = UI_LED_OFF;
			UILEDs.thumb_progress = UI_LED_OFF;
		}
		else
			memset(&UILEDs, UI_LED_OFF , sizeof(UILEDs));

      if (cTypeInfo[m_capType].statusLED == LED_LEFT)
			UILEDs.left_four_progress = LEDState;
		else if (cTypeInfo[m_capType].statusLED == LED_RIGHT)
			UILEDs.right_four_progress = LEDState;
		else if (cTypeInfo[m_capType].statusLED == LED_MID)
			UILEDs.thumb_progress = LEDState;
		r = TP_SetDeviceUI(m_hDev,UI_4_4_2_TRI_LED,&UILEDs);
	}
	return r;
}

//-----------------------------------------------------
//		SetPreview
//			Set up and begin Preview capture mode
//-----------------------------------------------------
long CCaptureDlg::SetPreview(void)
{
	long r;
	int dMode;
   RECT rc;
	unsigned short maxVideoW,maxVideoH;
   enum eCaptureType capType = cTypeInfo[m_capType].base_ctype;

	m_crColor = RGB(255,255,255);	// Display in White
   m_captureState = CSTATE_PREVIEW;
   UpdateDisplay(m_captureState,false);

      // Finger Text & Image - Pane 2
	m_ctlFngrTxt->SetWindowText(capDsp.fngrTxt);
   capDsp.fngrImg.DeleteObject();
	capDsp.fngrImg.LoadBitmap(MAKEINTRESOURCE(capDsp.fngrImgRes));
	m_ctlFngrImg->SetBitmap((HBITMAP)capDsp.fngrImg.GetSafeHandle());

		// Determine if desired size is acceptable to device
	if ((r = TP_GetMaxVideoSize(m_hDev,capType,&maxVideoW,
      &maxVideoH)) != TPAPI_NOERROR) return(r);

   wImage.GetClientRect(&rc);
	if (m_capType == CTYPE_HAND) 
   {
      YSize = rc.bottom - rc.top;
      if (YSize > maxVideoH) YSize = maxVideoH;

	   XSize = ((maxVideoW * YSize) / maxVideoH) & ~0x03;
	   if (XSize > maxVideoW) XSize = maxVideoW & ~0x03;
      dMode = DECORATED_HANDSPEED;
   }
	else 
   {
      XSize = (rc.right - rc.left) & ~0x03;
      if (XSize > maxVideoW) XSize = maxVideoW & ~0x03;

	   YSize = (maxVideoH * XSize) / maxVideoW;
	   if (YSize > maxVideoH) YSize = maxVideoH;
      dMode = UNDECORATED;
   }
   YSize = (YSize / BANDED_LINES) * BANDED_LINES;
	if (SetupBitmap(dMode,XSize,YSize,&m_bmImage,&bitmapData) 
			!= TPAPI_NOERROR) return(FALSE);

   CBitmap *pOld = dcMemImage.SelectObject(&m_bmImage);
   pOld->DeleteObject();

   OnUpdateDisplay(0,false);

	if ((r = SetUIProgress(m_capType, UI_LED_YELLOW, FALSE)) != TPAPI_NOERROR)
		return(r);

#ifdef DISPLAY_FPS
   numFrms = 0;
#endif
		// Begin Preview
   m_bCapturing = false;
	return(TP_StartPreview(m_hDev,m_capType,m_captureMode,
      XSize,YSize,ImgCallbk,this));
}

//----------------------------------------------------------
//		EvaluateRTQAInfo
//			Evaluate and display Real-time Quality Analysis 
//			information, SCAN/SAVE buttons, and return 
//			Accept/Reject
//----------------------------------------------------------
long CCaptureDlg::EvaluateRTQAInfo(void)
{
	long r;
	struct s_imgQA pParams;
   enum eCaptureType capType = cTypeInfo[m_capType].base_ctype;

	if ((r = TP_GetImageQA(m_hDev,&pParams)) == TPAPI_NOERROR)
	{		
//		if (m_RTQAThrhld > 0)
      if (pParams.good_area >= m_RTQAThrhld)
		//-------------------------------------
		//				Roll QA Info
		//-------------------------------------
		if (capType == CTYPE_ROLL) 
		{
			if (pParams.cover_factor < MIN_COVERAGE)
			{
				sprintf(capDsp.operTxt,"ROLL WIDER\n\n"
					"Coverage: %d\nRequired: %d",
					pParams.cover_factor,MIN_COVERAGE);
				r = TPAPI_UNKNOWN;
			}
			else if ((pParams.roll_time > 0) &&
				(pParams.roll_time < MIN_ROLLTIME))
			{
				sprintf(capDsp.operTxt,"ROLL SLOWER\n\n"
					"Roll Time: %d mS\nRequired: %d mS",
					pParams.roll_time,MIN_ROLLTIME);
				r = TPAPI_UNKNOWN;
			}
			else if (pParams.roll_time == 0)
			{
				strcpy(capDsp.operTxt,"STOP SLIPPING\n\n");
				r = TPAPI_UNKNOWN;
			}
		}
		else if (capType == CTYPE_HAND)
		{
			if (pParams.cover_factor < MIN_COVERAGE_HAND)
			{
				sprintf(capDsp.operTxt,"ROLL LONGER\n\n"
					"Coverage: %d\nRequired: %d",
					pParams.cover_factor,MIN_COVERAGE_HAND);
				r = TPAPI_UNKNOWN;
			}
		}
		//-------------------------------------
		//				"Good" Area Info
		//-------------------------------------
		if (r == TPAPI_NOERROR)
		{
			if (pParams.good_area < m_RTQAThrhld)
			{
				if ((pParams.light_area >= pParams.dark_area)
					&& (pParams.light_area >= pParams.smear_area))
					strcpy(capDsp.operTxt,"PRESS HARDER\n\n");
				else if ((pParams.dark_area > pParams.light_area)
					&& (pParams.dark_area > pParams.smear_area))
					strcpy(capDsp.operTxt,"LESS PRESSURE\n\n");
				else
					strcpy(capDsp.operTxt,"STOP SLIPPING\n\n");

				r = TPAPI_UNKNOWN;
			}
			else strcpy(capDsp.operTxt,"\n\n");
			sprintf(capDsp.operTxt,"%s"
					"Good Area: %d%%\nRequired: %d%%",
					capDsp.operTxt,pParams.good_area,m_RTQAThrhld);
		}
	}
			// Display Score
	capDsp.operState.DeleteObject();
	capDsp.btns.DeleteObject();
	if (r == TPAPI_NOERROR)
	{
		m_crColor = RGB(0, 255, 0);	// Display Score in Green
   	capDsp.operStateRes = IDB_P3_PASS;
    	capDsp.btnsRes = IDB_P4_REVIEW;
	}
	else
	{
		m_crColor = RGB(255, 0, 0);	// Display Score in Red
   	capDsp.operStateRes = IDB_P3_REJECT;
    	capDsp.btnsRes = IDB_P4_REJECT;
	}
	return(r);
}

//-----------------------------------------------------
//		CallbkCaptured
//			Get and display results of RTQA, prepare
//			operator for Review stage
//-----------------------------------------------------
long CCaptureDlg::CallbkCaptured(LPBYTE ImageBuf,
             unsigned short Xsize,unsigned short Ysize)
{
	long r = TPAPI_NOERROR;

	UpdateDisplay(CSTATE_CAPTURED,false);

	if (ImageBuf == NULL)		// no RTQA image
	{
		if (m_captureMode & CMODE_REVIEW)
		{
			strcpy(capDsp.operTxt,"\nREVIEW IMAGE");
         capDsp.btnsRes = IDB_P4_REVIEW;
		}
	}
	else
	{		// Display decorated image
		int dMode;
		if (m_capType == CTYPE_HAND) 
         dMode = DECORATED_RTQA|DECORATED_HANDSPEED;
		else dMode = DECORATED_RTQA;

      RGBQUAD pColors[256];
      int retVal;
      if ((retVal = GetDIBColorTable(dcMemImage,0,256,pColors)) == 256)
      {     // change to decorated palette
         SetupPalette(dMode,pColors, 256);
         if (SetDIBColorTable(dcMemImage,0,256,pColors) == 256)
         {
			   XSize = Xsize;
			   YSize = Ysize;
               // copy image
			   memcpy(bitmapData,ImageBuf,Xsize*Ysize);
			   wImage.Invalidate();
         }
      }
	}
	if (m_captureMode & CMODE_RTQA)
		r = EvaluateRTQAInfo();

		// Update display from dispatch thread
	PostMessage(WM_USER_DSPLY);
	return(r);
}
